/**
 * Kiubi Components
 *
 * Copyright 2023 Kiubi
 */
"use strict";

window.KiubiComponent=window.KiubiComponent||{};

KiubiComponent.ProductListing = function(options) {

    var productTpl = _.template(options.uiTpl.html());
    var basePrice;
    var lastMeta;
    var nb;
    var filters = {};
    var lock = false;
    var pagination = 24;
    var order = 'name';

    function reset() {
        basePrice = 'TTC';
        lastMeta = null;
        nb = 0;
        options.uiBtn.hide();
        options.uiList.html('');
    }

    // Genere le HTML pour un produit
    function renderProduct(data, nb) {

        // TODO data.header.length ?

        data.rate_5 = Math.round(data.rate/2);
        if (basePrice == 'TTC') {
            data.price = (data.price_max_ex_vat == data.price_min_ex_vat) ? data.price_min_inc_vat_label : ('' + data.price_min_inc_vat_label + ' - ' + data.price_max_inc_vat_label)
            data.base_price = data.price_base_inc_vat_label ? data.price_base_inc_vat_label : '';
        } else {
            data.price = (data.price_max_ex_vat == data.price_min_ex_vat) ? data.price_min_ex_vat_label : ('' + data.price_min_ex_vat_label + ' - ' + data.price_max_ex_vat_label)
            data.base_price = data.price_base_ex_vat_label ? data.price_base_ex_vat_label : '';
        }
        data.delay = ((nb%3)+1)*100;

        return productTpl(data);
    }

    // Process
    function processPage(meta, data){
        lock = false;
        if (meta.base_price) {
            basePrice = meta.base_price;
        }

        if (meta.items_count == 0) {
            options.uiListCount.html('Aucun r&eacute;sultat');
        } else {
            options.uiListCount.html('<span class="text-primary">' + meta.items_count + '</span>' + (meta.items_count == 1 ? ' produit' : ' produits'));
        }

        lastMeta = meta;
        var lastNb = nb;

        var html = _.reduce(data, function(acc, product){
            return acc + renderProduct(product, nb++);
        }, '');

        if (nb == 0) {
            options.uiBtn.hide();
            options.uiList.html('<div class="col-12"><p>Pas de produit disponible.</p></div>');
            return;
        }

        if (lastNb == 0) {
            options.uiList.html(html);
        } else {
            options.uiList.append(html);
        }

        if (kiubi.hasNextPage(meta)) {
            options.uiBtn.show();
        } else {
            options.uiBtn.hide();
        }

    }

    // Gere les erreurs API
    function processError(meta, error) {
        lock = false;
        options.uiList.html('<div class="col-12"><p class="text-danger">'+error.message+'</p></div>');
    }

    // Initialise les comportements
    this.init = function() {

        if (options.uiPagination.length) {
            pagination = options.uiPagination.first().data('pagination');
            options.uiPagination.first().removeClass('text-muted');
        }

        if (options.uiOrder.length) order = options.uiOrder.first().data('tri');

        options.uiBtn.on('click', function() {
            if (kiubi.hasNextPage(lastMeta)) {
                kiubi.getNextPage(lastMeta).done(processPage).fail(processError);
            }
            return false;
        });

        // Filtres :
        var $price = $("[data-filter='price'] input");
        var $tags = $("[data-filter='tags'] input");
        var $categ = $("[data-filter='category'] input");

        var that = this;

        $price.on('keyup', _.debounce(function(){
            var v = this.value.replace(',', '.');
            if (v !== '') {
                v = parseFloat(v);
                if (isNaN(v)) {
                    return;
                }
            }

            var fName = (this.name == 'pmin') ? 'price_min' : 'price_max';
            if (filters[fName] === v) return;
            filters[fName] = v;

            that.start();
        }, 200));

        $tags.on('change', function() {
            var t = [];
            $tags.filter(':checked').each(function(i, input){
                t.push($(input).data('tag'));
            });
            if (t == filters.tags) return;
            filters.tags = t;

            that.start();
        });

        $categ.on('change', function() {

            var t;
            if ($(this).data('categorie') === 'all') {
                $categ.filter(':checked').prop( "checked", false);
                $(this).prop( "checked", true);
                t = null;
            } else {
                t = [];
                $categ.filter('[data-categorie="all"]:checked').prop( "checked", false);
                $categ.filter(':checked').each(function(i, input){
                    t.push($(input).data('categorie'));
                });
            }

            if (t == filters.category_id) return;
            filters.category_id = t;

            that.start();
        });
		
		// Init categ filter state
		var t = [];
		if ($categ.length) {
			$categ.filter(':checked').each(function(i, input){
				var c = $(input).data('categorie');
				if (c == 'all' || !c) return;
				t.push(c);
			});
		} else {
			$("[data-filter='category'] a.active").each(function(i, link){
				var c = $(link).data('categorie');
				if (!c) return;
				t.push(c);
			});	
		}
		if (t.length > 0) filters.category_id = t;

        // Tri
        options.uiPagination.on('click', function(){
            pagination = $(this).data('pagination');
            options.uiPagination.addClass('text-muted');
            $(this).removeClass('text-muted');

            that.start();
            return false;
        });

        options.uiOrder.on('click', function(){
            order = $(this).data('tri');

            // Fermeture manuelle du dropdown, empechee par le return false
            $(this).parent().dropdown('toggle');

            that.start();
            return false;
        });

    };

    // Lance la recherche
    this.start = function() {
        if (lock) return;

        reset();

        var options = _.extend({
            limit: pagination,
            sort: order,
            extra_fields: 'price_label'
        }, filters);

        // Requete API
        lock = true;
        kiubi.search.products('', options).done(processPage).fail(processError);

    };

};

KiubiComponent.remoteGallery = function(options) {

    if (!options.containerEl || !options.containerEl.length) {
        console.log('Conteneur introuvable');
        return;
    }

    if (!options.uiTpl || !options.uiTpl.length) {
        console.log('Template introuvable');
        return;
    }

    if (!kiubi || !kiubi.media) {
        console.log('Client API Kiubi introuvable');
        return;
    }

    var $containerEl = options.containerEl;
    var imageTpl = _.template(options.uiTpl.html());

    kiubi.media.getFiles(options.folder_id, {
        extra_fields: 'thumb'
    }).done(function(meta,data) {

        var gallery = _.reduce(data, function(acc, file){
            if (file.type !== "image") {
                return acc;
            }
            acc += imageTpl({
                url : file.thumb.url,
                vignette : file.thumb.url_vignette,
                name: file.name,
                description: file.description
            });
            return acc;
        }, '');

        $containerEl.append(gallery);
    });

};

KiubiComponent.cartAdd = function(options) {

    var $addBtn = options.addBtn;
    var $cartSummary = options.cartSummary;
    var $cartNotification = options.cartNotification;

    var lock = false;

    function updateCart(cart,base_price, cartPayload) {

        // Cart summary update
        $('[data-cart-count]', $cartSummary).html(cart.items_count);
        $('[data-cart-total]', $cartSummary).html('article' + (cart.items_count > 1 ? 's' : '') + ' / ' + (base_price === 'TTC' ? cart.price_total_inc_vat_label : cart.price_total_ex_vat_label));
        $cartSummary.addClass('text-success');
        setTimeout(function() {
            $cartSummary.removeClass('text-success');
        }, 2000);

        // Cart notification
        if(cartPayload) {
            $('[data-cart-alert-desc]', $cartNotification).html(''+ ( cartPayload.qt>1 ? cartPayload.qt + ' x ' : '') + cartPayload.product.name  + ' - ' +  (base_price === 'TTC' ? cartPayload.variant.price_inc_vat_label  : cartPayload.variant.price_ex_vat_label));
            $cartNotification.removeClass('invisible').addClass('show');
            setTimeout(function() {
                $cartNotification.removeClass('show');
            }, 5000);
            setTimeout(function() {
                $cartNotification.addClass('invisible');
            }, 5200);
        }
    }

    $addBtn.on('click', function(e) {
        e.preventDefault();
        if (lock === true) return;
        lock = true;

        var pid, vid, qt = '+1';
        var $this = $(this);

        var f = $(this.form).serializeArray();
        $(f).each(function() {
            if (this.name === 'pid') pid = parseInt(this.value);
            if (this.name === 'vid') vid = parseInt(this.value);
            if (this.name === 'qt')  qt = '+' + parseInt(this.value);
        });

        $('#' + $this.data('cart-error')).html('');

        var promise = (pid) ? kiubi.catalog.getProduct(pid, {extra_fields:'price_label'}) : $.Deferred().reject();
        promise.then(function(meta,product) {

            if (!vid) vid = product.variants[0].id;
            var cartPayload = {
                product: product,
                variant:_.findWhere(product.variants, {id:vid}),
                qt:parseInt(qt)
            };
            return kiubi.cart.addItem(vid, qt, 'strict', {
                extra_fields: 'price_label'
            }).done(function(meta,data) {
                if (data.cart) {
                    updateCart(data.cart,meta.base_price || 'TTC', cartPayload);
                }
                $this.removeClass('btn-warning').addClass('btn-success');
                setTimeout(function() {
                    $this.removeClass('btn-success').addClass('btn-warning');
                }, 2000);
            });
        }).fail(function(meta,error) {
            $('#' + $this.data('cart-error')).html('<i class="fe fe-alert-triangle mr-2"></i>' + error.message);
        }).always(function(){
            lock = false;
        });

    });

};


KiubiComponent.schedulingPicker = function() {

        // Parametrage du module
        var carriersSetup = [];
        var isLoaded = false;

        /**
         * Charge les regles des transporteurs
         *
         * @returns {jQuery.Deferred}
         */
        function loadSetup() {
            if (isLoaded) {
                return $.Deferred().resolve(carriersSetup);
            }

            return kiubi.cart.getCarriers().then(function(meta, carriers){

                $(carriers).each(function(){
                    if(!this.require_scheduling || this.require_scheduling == 'no') return;

                    carriersSetup.push({
                        id:this.id,
                        rules:this.scheduling_rules,
                        require_scheduling:this.require_scheduling
                    });
                });

                isLoaded = true;

                return $.Deferred().resolve(carriersSetup);
            },function(){
                console.warn('Erreur API - impossible de charger la liste des transporteurs');
            });
        }

        /**
         * Retourne la liste des plages horaires.
         *
         * @param {Object} setup
         * @param {Number} dow
         * @returns {Array}
         */
        function timeFrames(setup, dow) {

            var hours = [];

            // Enable All if not set
            if(!setup.open_days[dow].time_frames || setup.open_days[dow].time_frames.length == 0) {
                for(var i=0; i<=23; i++) {
                    hours.push(i);
                }
                return hours;
            }

            // Disable all
            hours.fill(false, 0, 23);
            $(setup.open_days[dow].time_frames).each(function(){
                var interval = this.split('-');
                var min = parseInt(interval[0]);
                var max = parseInt(interval[1]);
                for(var i=min; i<=max; i++) {
                    hours.push(i);
                }
            });

            return hours;
        }

        // Init
        this.init = function() {

            loadSetup().done(function(carriersSetup){

                var now = moment();

                $(carriersSetup).each(function(){

                    var rules = this.rules;
                    var id = this.id;

                    var closed = rules.closed_days.map(function(day){
                        return moment(day, "DD/MM/YYYY");
                    });

                    var closed_dow = rules.open_days.reduce(function(list, day){
                        if(!day.is_open) list.push(day.day_of_week);
                        return list;
                    }, []);

                    // Min interval si time > time shift
                    var i_min = rules.scheduling_interval_min;
                    if(rules.limit_hour && ''+now.hour()+':'+now.minute() > rules.limit_hour) {
                        i_min++;
                    }

                    var $date_picker = $('#transporteur_date_'+id);

                    var params = {
                        locale: 'fr',
                        format: "DD/MM/YYYY",
                        disabledDates: closed,
                        daysOfWeekDisabled: closed_dow,
                        minDate: rules.scheduling_interval_min ? moment().startOf('day').add(i_min, 'd') : moment().startOf('day'),
                        maxDate: rules.scheduling_interval_max ? moment().startOf('day').add(rules.scheduling_interval_max, 'd') : false,
                        useCurrent: false
                    };

                    $date_picker.datetimepicker(params);
                    $date_picker.on("dp.change", function (e) {
                        $('#transporteur_'+id).prop('checked', true);
                    });

                    if(this.require_scheduling == 'datetime') {

                        var $time_picker = $('#transporteur_heure_'+id).datetimepicker({
                            format: "HH",
                            useCurrent: true
                        });

                        $date_picker.on("dp.change", function (e) {
                            $time_picker.data("DateTimePicker").enabledHours(timeFrames(rules, e.date.day()));
                        });
                    }
                });

            });
        };
};
